# -*- coding: utf-8 -*-

"""
 |---------------------------------------------------------------------------|
 | KSI 2014/2015                                                             |
 | Úloha 2-2                                                                 |
 | Jan Horáček                                                               |
 | Gymnázium, Brno, Vídeňská 47                                              |
 | jan.horacek@seznam.cz                                                     |
 | 14.12.2014                                                                |
 -----------------------------------------------------------------------------

Hezký den, orgové.
V této úloze jsem se, jako v jedné z mála, rozhodl neodevzdávat řešení v podobě
dokumentu, ale pouze ve formě zdrojového kódu. O to více jsem do zdrojáku psal
komentářů, takže pevně věřím, že nebudete o žádný popis ochuzeni.

--------------------------------------------------------------------------------

V tomto priklade budete mat za ulohu zmodifikovat nasledujuci kod 
reprezentujuci baliacu linku tak, aby fungoval efektivnejsie.

Cely system sa sklada s niekolkych casti:

1. Baliaci system (packer) - ktory je reprezentovany frontou do ktorej
su v nahodnych intervaloch generovane nove darceky, z ktorej sa postupne
beru darceky do druhej casti systemu.

2. Prevoznik prepravok(containers - ContainerSystem) obsahuje prepravky,
do ktorych na seba naskladavame darceky, ktore vychadzaju z baliaceho systemu
Kapacita prepravky je definovna v globalnej premennej(systems.CONTAINER_CAPACITY)

3. Transport (transfer) - ktory na opacnom konci prevoznika odobera
z prepravok darceky a posiela ich prec.

Vasou ulohou bude zaistit nasledujuce poziadavky na system:
1. system nepusti poskodeny darcek na prepravu(transfer) - k tomuto
ucelu bude sluzit metoda packer.ReturnPresent(present),ktora vrati darcek
na prebalenie
2. system sa bude snazit minimalizovat pocet prepraviek v obehu, zaroven
ale bude klast poziadavku na to, ze nakladanie a vykladanie balikov do
prepraviek nebude musiet cakat.
3. po skonceni behu systemu chceme aby sa vsetky uspesne zabalene darceky
dostali az na transport.

Mozu sa vam hodit nasledujuce informacie o triedach:

Trieda Present, reprezentujuca darcek a obsahuje:
atribut "broken", teda pomocou present.broken dokazete zistit ci je darcek poskodeny
atribut "present_id", moze sluzit pre identifikaciu darceku

Trieda Container, reprezentujuca prepravku a obsahuje:
zasobnik darcekov
atribut "next", ukazatel na nasledujucu prepravku
atribut "prev", ukazatel na predchadzajucu prepravku
metodu PushPresent(present) - vlozi darcek do prepravky
metodu PopPresent() - vrati darcek na vrchu prepravky
metodu Size() - vrati pocet prvkov v prepravke
metodu IsEmpty() - vrati, ci je prepravka prazdna
metodu HasSpace() - vrati, ci sa v prepravke este nachadza miesto

Trieda systems.Packer reprezentuje system, ktory generuje zabalene darceky:
metoda HasPresent() - vrati ci obsahuje nejaky zabaleny darcek
metoda GetPresent() - vrati darcek a odobere ho z packera
metoda ReturnPresent(present) - je mozne vyuzit k navratu darceka do
baliaceho na prebalenie

Trieda systems.Transfer, reprezentuje prepravnik z baliaceho systemu:
metoda SetContainer() - nastavi z ktorej prepravky sa odoberaju darceky

POZOR Container nekontroluje ci do neho vkladate prvky navyse v takom
pripade darceky su nenavratne stratene.
"""

import binary2_2 as systems
import time

#DEBUG sluzi ku kontrolnym vypisom, ak ho nastavite na True
DEBUG = False

"""
Trieda definujuca system s prepravkami

Reprezentuje zretazeny cyklicky zoznam, kde kazda prepravka(container)
ma ukazatel na predchadzajuci a nasledujuci kontainer v baliacej linke.

Uchovava si ukazatele na prvu(first) a poslednu prepravku(last). Kde do
prvej prepravky system naklada a z poslednej vyklada darceky.

Nakladanie funguje tak, ze system do prepravky frst naklada kym nie je
plna, inak caka kym sa uvolni prepravka first. Pri vypnuti systemu sa
generovanie darcekov zastavuje.

Vykladanie prebieha tak ze transfer odobera darceky dokym su k dispozicii
z nastavenej prepravky
"""
class ContainerSystem:

    """
    Pri inicializaci se vytvori list kontejneru, ktery v pocatku obsahuje
    INIT_CONT_CNT kontejneru. Je dobre tuto konstantu ponechat na cisle 2,
    protoze optimalizace v programu by mohly vyhodnotit zbytecne prepravky
    a mazat je (coz dle zadani trva dlouho).

    Tento list je vniman jako cyklicky seznam tak, ze napriklad preskok
    na dalsi polozku se provadi takto:
      dalsi = (aktualni+1) % len(self.containers.conts)

    Do tohoto seznamu se postupne pridavaji a odebiraji kontejnery (podle
    potreby).

    """
    def __init__(self):
        INIT_CONT_CNT = 2

        if(DEBUG): print("Init container list")

        self.conts = [];
        for i in range(INIT_CONT_CNT):
            self.conts.append(systems.Container())

    """
    Tady jsou definaovane "ukazatele" na aktualne nakladajici a aktualne
    vykladajici prepravku. Na zacatku nakladame do 0. prepravky a z zadne
    nevykladame (vykladame z 0. prepravky az po jejim naplneni).

    Nakladka je vzdy index prepravky (i v pripade, ze uz je naklad ukoncen).

    """
    nakladka = 0
    vykladka = -1        # Pokud je vykladka -1, je neni zadna paleta vykladana


    """
    Priklad stavu systemu pro 5 prepravek:
      prepravka [0] - naklada se
      prepravka [1] - je nalozena, ceka na vykladku
      prepravka [2] - je nalozena, ceka na vykladku
      prepravka [3] - vyklada se
      prepravka [4] - je vylozena, ceka na nakladku

      prepravka [5] = prepravka [0] atd. (cyklicky list)

      V tomto pripade je nakladka = 0 a vykladka = 3
    """


    """
    Prida novou prepravku do LISTu prepravek na stanovene misto.
    """
    def AddContainer(self, index):
        self.conts.insert(index, systems.Container())

    """
    Maze zadanou prepravku z LISTu prepravek.
    """
    def RemoveContainer(self, container):
        self.conts.remove(container)


"""
Trieda system sluzi na reprezentaciu celeho systemu.
"""
class System:
    """
    Inicializacia vsetkych komponent systemu
    """
    def __init__(self):
        if(DEBUG): print("Init system")
        self.packer = systems.Packer()
        self.containers = ContainerSystem()
        self.transfer = systems.Transfer()
        self.run = False
        self.ended = False

    """
    Spustenie balenia a transferu
    
    Nemodifikujte nasledujucu metodu
    """
    def Start(self):
        if(DEBUG): print("Running system")
        self.run = True
        self.packer.RunPacker()
        self.transfer.RunTransfer()

    """
    Primarni funkce systemu:
    Je zavolana jednou a pak ceka na ukonceni (self.run)
    """
    def RunContainerSystem(self):
        """
        System bezi, pokud je zvnejsku zaply, ci pokud je vykladana
        nejaka prepravka.

        Kodem nize je zaruceno to, ze pokud v systemu existuje alespon jeden
        darek, je tento darek vykladan
        (tj. v systemu nemuzou zustat visici darky na konci).

        Zakladni princip: volat metody UpdateNakladka a UpdateVykladka, ktere
        se staraji o jednotlive cinnosti.
        """
        while (self.containers.vykladka > -1) or (self.run):
            self.UpdateNakladka()
            self.UpdateVykladka(self.run)

        #zastavenie systemu nemente!
        self.ended = True
        self.transfer.StopTransfer()

    """
    Tato metoda obsluhuje balici system - bere z nej darky, dava je do
    prepravek, odesila prepravky k zabaleni a vytvari dalsi prepravky v pripade,
    ze je jich nedostatek.

    Pozn.: Aktualne nakladany balik je v self.containers.nakladka.
    """
    def UpdateNakladka(self):
      # Je na balici lince zabaleny darek ?
      if self.packer.HasPresent():
        # Ano -> ziskame ho.
        present = self.packer.GetPresent()

        # Je darek poskozeny?
        if present.broken:
          # Ano -> vratit a ukoncit provadeni metody
          self.packer.ReturnPresent(present)
          return

        # Darek neni poskozeny -> je k dispozici kontejner na tento darek ?
        # Pokud se aktulni kontejner (do ktereho by mel prijit darek) vyklada,
        #   nebo nema misto, je vytvoren kotejner dalsi na aktulnim indexu
        if (self.containers.nakladka == self.containers.vykladka) or (not self.containers.conts[self.containers.nakladka].HasSpace()):
          # Kontejner, ktery je urcen k nakladani, se jeste vyklada -> je potreba pridat kontejner na index self.nakladka (ten bude logicky prazdny)
          self.containers.AddContainer(self.containers.nakladka)
          # Pokud se nejaky kontejner vyklada, posuneme index vykladaciho kontejneru
          # Tuto podminku si muzeme dovolit, protoze index_noveho_kotejneru <
          #  self.containers.vykladka vzdy !
          # Tj. nove kontejnery pridavame na zacatek listu, ne na jeho konec.
          if self.containers.vykladka > -1: self.containers.vykladka = (self.containers.vykladka+1) % len(self.containers.conts)
          if(DEBUG): print "<<<< Pridavam kontejner, celkem kontejneru: ", len(self.containers.conts)

        # Mame volnou prepravku -> nalozime darek do prepravky
        if(DEBUG): print "Pridavam darek do kontejneru ", self.containers.nakladka, self.containers.conts[self.containers.nakladka]
        self.containers.conts[self.containers.nakladka].PushPresent(present)
        if not self.containers.conts[self.containers.nakladka].HasSpace():
          # Pokud je prepravka plna, posleme ji do fronty k vykladaci lince.
          if(DEBUG): print "---- Nakladaci prepravka plna"

          # Pokud se nic nevyklada, zacneme aktualne nalozenou prepravku
          #   ihned vykladat.
          if (self.containers.vykladka < 0):
            self.containers.vykladka = self.containers.nakladka;
            self.transfer.SetContainer(self.containers.conts[self.containers.vykladka])
            if(DEBUG): print "!!! Vykladam kontejner ", self.containers.vykladka

          # Pristi darek tedy budu nakladat uz do dalsi prepravky.
          self.containers.nakladka = (self.containers.nakladka+1) % len(self.containers.conts)


    """
    Tato metoda obsluhuje system, ktery vyklada darky z prepravek - kontroluje,
    jestli ma nejake prepravky k vylozeni, vyklada je, kontroluje, kdy jsou
    vylozeny a dava je do fronty pro nakladku darku.

    Pozn.: Aktualne vykladany balik je v self.containers.vykladka
    """
    def UpdateVykladka(self, run):
      # Je nejaka prepravka k vylozeni?
      if self.containers.vykladka > -1:
        if(DEBUG): print "++++ Vykladam kontejner", self.containers.vykladka

        # Je aktualne vykladane prepravka uz vylozena?
        if self.containers.conts[self.containers.vykladka].IsEmpty():
          # Ano
          if(DEBUG): print "++++ Vykladany kontejner je prazdny", self.containers.vykladka

          # Kontrola prebytku prepravek v systemu
          self.KontrolujPrebytek(self.containers.vykladka)

          # Ceka nejaky dalsi kontejner ve vstupni fronte pro vykladku?
          self.containers.vykladka = (self.containers.vykladka+1) % len(self.containers.conts);
          if(DEBUG): print "-------------------------- Kontroluji obsah kontejneru", self.containers.vykladka, self.containers.conts[self.containers.vykladka]

          if (not self.containers.conts[self.containers.vykladka].IsEmpty()) and ((self.containers.vykladka != self.containers.nakladka) or (not run)):
            # Ano, ceka na nas neprazdny kontejner, ktery se uz nenaklada.
            # Sem se take skoci, pokud nas ceka kontejner, ktery se jeste
            #  naklada, ale uz je vyply system
            #  (to je proto, ze tento kontejner je potreba vylozit nezavisle na
            #   tom, koli je v nem darku - protoze linka se ukoncuje)
            if(DEBUG): print "++++ Vykladam novy kontejner", self.containers.vykladka

            # -> spusti vybalovani darku z kontejneru.
            self.transfer.SetContainer(self.containers.conts[self.containers.vykladka])
          else:
            # Pokud na vstupu neceka zadna prepravka na vybaleni,
            #   cekame, az nejaka na vstupu bude.
            if(DEBUG): print "++++ Vsechny kontejnery vylozeny"
            self.containers.vykladka = -1;



    """
    Tato funkce kontroluje prebytek prepravek v systemu.
    Je volana (logicky) pri mazani prepravek ze systemu.

    Prebytek nastava, pokud aktualni prepravka (parametr "vylozen")
    a prepravka prede mnou jsou prazdne (a nenaklada se do ni).
    System si tedy udrzuje jednu rezervni prepravku, proto po zastaveni cele
    linky skonci prave s jednou prepravkou.
      Tuto technologii jsem zvolil za ucelem minimalizace mazani a pridavani
      prepravek.
    """

    def KontrolujPrebytek(self, vylozen):
      # Zjisteni prepravky, ktera je "prede mnou"
      kontrola = (vylozen+len(self.containers.conts)-1) % len(self.containers.conts)

      # Je tato prepravka prazdna a nenaklada se do ni?
      if (kontrola != self.containers.nakladka) and (kontrola != self.containers.vykladka) and (self.containers.conts[kontrola].IsEmpty()):
        # Ano -> odstranit
        if(DEBUG): print "Odstranuji kontejner", kontrola
        self.containers.RemoveContainer(self.containers.conts[kontrola])

        # Posunout indexy nakladky a vykladky
        if kontrola < self.containers.nakladka: self.containers.nakladka = (self.containers.nakladka+len(self.containers.conts)-1) % len(self.containers.conts);
        if kontrola < self.containers.vykladka: self.containers.vykladka = (self.containers.vykladka+len(self.containers.conts)-1) % len(self.containers.conts);

    """
    Zastavi beh systemu
    """
    def Stop(self):
        if(DEBUG): print("Stopping system")
        self.run = False
        self.packer.StopPacker()

        """
        Zde nasleduje osetreni situace, kdy doslo k tak brzkemu vypnuti, ze
        je v prvni prepravce nejake zbozi, ale tato prepravka jeste nebyla
        vyprazdnena.
        Pokud tato situace nastane, je potreba tuto prepravku nezavisle na
        poctu darku v ni vylozit na vystupni linku.

        Tento kod funguje i v pripade, ze po ukonceni programu touto metodou
        prijdou dalsi darky (musi ale prijit driv, nez v systemu bude 0 darku
        ve vsech prepravkach - pokud to nestihne, system se jednoduse ukonci,
        protoze v nem nebudou zadne darky).
        """
        if (self.containers.vykladka < 0) and (not self.containers.conts[self.containers.nakladka].IsEmpty()):
          self.containers.vykladka = self.containers.nakladka
          self.transfer.SetContainer(self.containers.conts[self.containers.vykladka])

          # Pokud by nahodoou po ukonceni prisly dalsi darky, dame je do
          #  nasledujici prepravky, pokud existuje
          # Pokud v systemu existuje jen jedna prepravka, program se o vytvoreni
          #  dalsi prepravky postara pri prichodu darku.
          if len(self.containers.conts) > 1:
            self.containers.nakladka = (self.containers.nakladka+1) % len(self.containers.conts)

    """
    Zisti ci system ukoncil pracu
    """
    def HasEnded(self):
        return self.ended
